//=============================================================================
// IndividualBuffRateStandard.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc スキルごとに指定した倍率で、標準の強化/弱体として効果を付与します。
 * @author あなたの名前 または ペンネーム
 * @url プラグインを公開する可能性のあるURL（任意）
 * @help IndividualBuffRateStandard.js
 *
 * このプラグインは、スキルのメモ欄に記述することで、指定した個別倍率と
 * ターン数を持つ強化/弱体効果を、標準の強化/弱体システム（アイコン表示、
 * 段階管理、ターン管理、相殺）を利用して付与します。
 *
 * ---使い方---
 * 強化/弱体効果を設定したいスキルのメモ欄に、以下のように記述します。
 *
 * <SetIndividualBuff paramId: id rate: r turns: t>
 *
 * ・id: 強化/弱体する能力値のIDを指定します。
 *   0: 最大HP / 1: 最大MP / 2: 攻撃力 / 3: 防御力
 *   4: 魔法力 / 5: 魔法防御 / 6: 敏捷性 / 7: 運
 * ・r: 強化/弱体倍率を指定します。
 *   (例: 1.3 で1.3倍強化, 0.8 で0.8倍弱体)
 * ・t: 効果の持続ターン数を指定します。
 *
 * 例1: 攻撃力(paramId=2)を5ターン、1.3倍にするスキル
 * <SetIndividualBuff paramId: 2 rate: 1.3 turns: 5>
 *
 * 例2: 防御力(paramId=3)を3ターン、0.8倍にするスキル (弱体)
 * <SetIndividualBuff paramId: 3 rate: 0.8 turns: 3>
 *
 * ---注意点---
 * ・このプラグインによる強化/弱体効果は、標準の強化/弱体として扱われます。
 *   アイコンが表示され、最大2段階まで累積し、強化と弱体は相殺されます。
 * ・能力値計算には、メモ欄で指定した最新の個別倍率が適用されます。
 *   (例: 1.3倍強化のスキル使用後、1.5倍強化スキルを使うと1.5倍になる)
 *   (例: 1.3倍強化中に、0.8倍弱体スキルを使うと、強化は1段階解除され、
 *     倍率は標準の弱体1段階(0.75倍)になります。個別倍率0.8倍は適用されません)
 * ・スキルの「使用効果」欄には何も設定しないでください。
 *   設定すると意図しない動作やメッセージ表示の原因になる可能性があります。
 * ・個別倍率は強化状態の時のみ適用されます。弱体状態の倍率はツクールMZ標準です。
 *   (弱体にも個別倍率を適用したい場合はプラグインの改造が必要です)
 *
 * ---利用規約---
 * 自由にご利用ください。
 * 改変、再配布も自由です。
 * クレジット表記は任意です。
 *
 * @command dummy
 * @text (コマンドなし)
 * @desc このプラグインにはプラグインコマンドはありません。
 */

(() => {
    'use strict';

    const pluginName = "IndividualBuffRateStandard";

    //-----------------------------------------------------------------------------
    // Game_BattlerBase
    //-----------------------------------------------------------------------------

    const _Game_BattlerBase_initMembers = Game_BattlerBase.prototype.initMembers;
    Game_BattlerBase.prototype.initMembers = function() {
        _Game_BattlerBase_initMembers.call(this);
        this.clearIndividualBuffRates();
    };

    // 個別強化倍率情報を初期化
    Game_BattlerBase.prototype.clearIndividualBuffRates = function() {
        // { paramId: rate }
        this._individualBuffRates = {};
    };

    // 全ての強化/弱体をクリアする際に個別倍率もクリア
    const _Game_BattlerBase_clearBuffs = Game_BattlerBase.prototype.clearBuffs;
    Game_BattlerBase.prototype.clearBuffs = function() {
        _Game_BattlerBase_clearBuffs.call(this);
        this.clearIndividualBuffRates();
    };

    // 特定の強化/弱体が解除される際に個別倍率もクリア
    const _Game_BattlerBase_eraseBuff = Game_BattlerBase.prototype.eraseBuff;
    Game_BattlerBase.prototype.eraseBuff = function(paramId) {
        // 元の処理で段階が 0 になるか確認するために先に実行
        const currentLevel = this._buffs[paramId];
        _Game_BattlerBase_eraseBuff.call(this, paramId);
        // バフ段階が0になった場合、またはデバフになった場合に個別倍率を削除
        if (this._buffs[paramId] <= 0 && currentLevel > 0) {
             delete this._individualBuffRates[paramId];
             console.log(`[${pluginName}] Erased individual buff rate for paramId: ${paramId}`);
        }
    };

    // 個別強化倍率を設定
    Game_BattlerBase.prototype.setIndividualBuffRate = function(paramId, rate) {
        this._individualBuffRates[paramId] = rate;
        console.log(`[${pluginName}] Set individual buff rate for paramId: ${paramId} to ${rate}`);
    };

    // 個別強化倍率を取得
    Game_BattlerBase.prototype.individualBuffRate = function(paramId) {
        // 強化状態の場合のみ個別倍率を返す (this._buffs[paramId] > 0)
        return this._buffs[paramId] > 0 ? (this._individualBuffRates[paramId] || 1.0) : 1.0;
    };

    // 能力値計算処理をオーバーライド
    const _Game_BattlerBase_param = Game_BattlerBase.prototype.param;
    Game_BattlerBase.prototype.param = function(paramId) {
        const baseValue = this.paramBase(paramId);
        const plusValue = this.paramPlus(paramId);
        const rate = this.paramRate(paramId); // 特徴による割合補正
        let buffRate = 1.0;

        // 標準の強化/弱体段階を取得
        const buffLevel = this._buffs[paramId] || 0;

        if (buffLevel > 0) {
            // 強化状態の場合
            const individualRate = this.individualBuffRate(paramId);
            if (individualRate !== 1.0) {
                // 個別倍率が設定されていればそれを使用
                buffRate = individualRate;
            } else {
                // 個別倍率がなければ、標準の強化倍率を使用 (コアスクリプトの計算を利用)
                // ただし、標準の paramBuffRate は個別倍率を考慮しないため、ここで再計算
                buffRate = 1.0 + buffLevel * 0.25; // MZ標準の強化倍率
            }
        } else if (buffLevel < 0) {
            // 弱体状態の場合、標準の弱体倍率を使用 (個別倍率は弱体には適用しない仕様)
            buffRate = Math.max(1.0 + buffLevel * 0.25, 0); // MZ標準の弱体倍率
        }

        // 最終的な能力値を計算
        let finalValue = (baseValue + plusValue) * rate * buffRate;

        // 最大MP(paramId=1)以外は整数化
        if (paramId !== 1) {
            finalValue = Math.floor(finalValue);
        }

        const maxValue = this.paramMax(paramId);
        const minValue = this.paramMin(paramId);
        return Math.round(finalValue.clamp(minValue, maxValue));
    };

    //-----------------------------------------------------------------------------
    // Game_Battler
    //-----------------------------------------------------------------------------

    // ターン経過処理を拡張して、効果切れ時に個別倍率もクリア
    const _Game_Battler_updateBuffTurns = Game_Battler.prototype.updateBuffTurns;
    Game_Battler.prototype.updateBuffTurns = function() {
        // 事前に現在のバフレベルを保持
        const buffsBefore = { ...this._buffs };

        _Game_Battler_updateBuffTurns.call(this); // 元のターン経過処理

        // ターン経過後にバフレベルが0以下になったか確認
        for (const paramId in buffsBefore) {
            if (buffsBefore[paramId] > 0 && this._buffs[paramId] <= 0) {
                delete this._individualBuffRates[paramId];
                 console.log(`[${pluginName}] Erased individual buff rate for paramId: ${paramId} due to turn end`);
            }
        }
    };

    //-----------------------------------------------------------------------------
    // Game_Action
    //-----------------------------------------------------------------------------

    const _Game_Action_apply = Game_Action.prototype.apply;
    Game_Action.prototype.apply = function(target) {
        const result = target.result(); // 事前に結果オブジェクトを取得
        _Game_Action_apply.call(this, target); // 元の処理（ダメージ計算など）

        // スキルであり、かつ効果が有効だった場合のみ処理
        if (this.isSkill() && result.isHit()) {
            const skill = this.item();
            const note = skill.note;

            // メモ欄から個別強化設定を読み取る
            const buffMatches = note.match(/<SetIndividualBuff\s+paramId:\s*(\d+)\s+rate:\s*([\d.]+)\s+turns:\s*(\d+)>/gi);

            if (buffMatches) {
                buffMatches.forEach(tag => {
                    const match = tag.match(/<SetIndividualBuff\s+paramId:\s*(\d+)\s+rate:\s*([\d.]+)\s+turns:\s*(\d+)>/i);
                    if (match) {
                        const paramId = parseInt(match[1], 10);
                        const rate = parseFloat(match[2]);
                        const turns = parseInt(match[3], 10);

                        if (!isNaN(paramId) && !isNaN(rate) && !isNaN(turns) && turns > 0) {
                            if (rate > 1.0) {
                                // 強化の場合
                                target.addBuff(paramId, turns); // 標準の強化を適用 (段階とターン)
                                // addBuff の後に個別倍率を設定
                                target.setIndividualBuffRate(paramId, rate);
                            } else if (rate < 1.0) {
                                // 弱体の場合 (標準の弱体として適用、個別倍率は適用しない)
                                target.addDebuff(paramId, turns);
                                // 弱体化によって既存の個別強化倍率が解除される場合があるのでクリア処理
                                if (target._buffs[paramId] <= 0) {
                                     delete target._individualBuffRates[paramId];
                                }
                            }
                            // rateが1.0の場合は何もしない
                        } else {
                            console.error(`[${pluginName}] Invalid tag format or values in Skill ID ${skill.id}: ${tag}`);
                        }
                    }
                });
                // このプラグインによる効果があれば、アクションは成功したとみなす
                result.success = true; // これにより「効果が無かった」メッセージを防ぐ
            }
        }
    };

})();
